#ifndef _DECODER_
#define _DECODER_
#pragma once

#include "ChartItem.h"
#include "Instance.h"
#include "common.h"
#include "Options.h"
using namespace egstra;

namespace dparser {

	class Decoder
	{
	public:
		Decoder() {}

		virtual ~Decoder(void) {}

		void decodeInterface( Instance *inst, const bool constrained ) {
			viterbi(inst, constrained);
			get_result(inst);
			dealloc();
		}

		void compute_marginals(const Instance * const inst, const bool constrained) {
			forward(inst, constrained);
			if (equal_to_negative_infinite(log_Z(inst))) {
				cerr << "log_Z() == DOUBLE_NEGATIVE_INFINITY!" << endl;
				exit(-1);
			}
			backward(inst, constrained);
			check_marginal_prob(inst);
		}

		double marginal_prob(const Instance * const inst, int i, int t) const {
			const double a = _forward_chart[i][t];
			const double b = _backward_chart[i][t];
			if (equal_to_negative_infinite(a) || equal_to_negative_infinite(b)) {
				return 0;
			} else {
				return exp(a + b  - log_Z(inst));
			}
		}
		double marginal_prob(const Instance * const inst, int i, int tL1, int t) const {
			const double a = _forward_chart[i-1][tL1];
			const double b = _backward_chart[i][t];
			if (equal_to_negative_infinite(a) || equal_to_negative_infinite(b)) {
				return 0;
			} else {
				return exp(a + b + inst->prob_bigram[i][tL1][t] + inst->prob_unigram[i][t] - log_Z(inst));
			}
		}

		double log_Z(const Instance * const inst) const {
			return _forward_chart[inst->size()][pos_id_dummy];
		}

		void process_options() {}
		
		virtual void use_marginal_as_arc_score(Instance * const inst);

	public:
		static int T;
		static int pos_id_dummy;

	protected:
		NRMat< const ChartItem * > _chart;		// N * T
		NRMat<double> _forward_chart;
		NRMat<double> _backward_chart;
	protected:
		virtual void forward(const Instance * const inst, const bool constrained);
		virtual void backward(const Instance * const inst, const bool constrained);
		virtual void viterbi(const Instance *inst, const bool constrained);

	protected:
		virtual void get_result(Instance *inst) const;
		virtual void get_best_parse_recursively( Instance *inst, const ChartItem * const item ) const;
		virtual void check_marginal_prob(const Instance * const inst);

		virtual void dealloc() {
			dealloc_m2(_chart);
			_forward_chart.dealloc();
			_backward_chart.dealloc();
		}

	protected:
		static void log_add_if_not_negative_infinite(double &self, const double a, const double b) {
			if (!equal_to_negative_infinite(a)) {
				assert(!equal_to_negative_infinite(b));
				if (equal_to_negative_infinite(self)) {
					self = a + b;
				} else {
					log_add_another(self, a + b);
				}
			}
		}
		static void log_add_if_not_negative_infinite(double &self, const double a, const double b, const double c) {
			if (!equal_to_negative_infinite(a) && !equal_to_negative_infinite(b)) {
				if(equal_to_negative_infinite(c)) {
					cerr << "log_add_if_not_negative_infinite(self, a, b, c): c = " << c << endl;
					exit(-1);
				}
				if (equal_to_negative_infinite(self)) {
					self = a + b + c;
				} else {
					log_add_another(self, a + b + c);
				}
			}
		}

		static void log_add_another(double &self, const double another) {
			self = log_add(self, another);
		}

		static double log_add(const double a, const double b)
		{
			if (a>b)
				return a+log(1+exp(b-a));
			else
				return b+log(1+exp(a-b));
		}

		bool add_item(const ChartItem * &add_place, const ChartItem * const new_item) {
			if (add_place == NULL) {
				add_place = new_item;
				return true;
			}
			if (add_place->_prob < new_item->_prob - EPS) { // absolutely less than the new item
				delete add_place;
				add_place = new_item;
				return true;
			} else {
				delete new_item;
				return false;
			}
		}
		
		static void dealloc_m2(NRMat< const ChartItem * > &_chart);
	};
	
	inline void Decoder::dealloc_m2(NRMat< const ChartItem * > &_chart) {
		const ChartItem  **m2 = _chart.c_buf();
		for (int i = 0; i < _chart.size(); ++i, ++m2) {
			const ChartItem * pitem = (*m2);
			if (pitem) delete pitem;			
		}
		_chart.dealloc();
	}
} // namespace dparser

#endif

